<?php
header('Content-Type: application/json');
require __DIR__ . '/../../vendor/autoload.php';

// Authentication check
if (!isset($_SERVER['HTTP_ADMIN_TOKEN']) || $_SERVER['HTTP_ADMIN_TOKEN'] !== $_ENV['ADMIN_SECRET']) {
    http_response_code(401);
    die(json_encode(['error' => 'Unauthorized']));
}

// Database connection
$db = new mysqli(
    $_ENV['DB_HOST'],
    $_ENV['DB_USER'],
    $_ENV['DB_PASSWORD'],
    $_ENV['DB_NAME']
);

if ($db->connect_error) {
    http_response_code(500);
    die(json_encode(['error' => 'Database connection failed']));
}

try {
    // Get filters from request
    $params = [
        'page' => max(1, $_GET['page'] ?? 1),
        'limit' => min(100, $_GET['limit'] ?? 20),
        'status' => $_GET['status'] ?? null,
        'date_from' => $_GET['date_from'] ?? null,
        'date_to' => $_GET['date_to'] ?? null,
        'search' => $_GET['search'] ?? null
    ];

    // Build SQL query
    $where = [];
    $bindTypes = '';
    $bindValues = [];
    
    if ($params['status']) {
        $where[] = "status = ?";
        $bindTypes .= 's';
        $bindValues[] = $params['status'];
    }
    
    if ($params['date_from']) {
        $where[] = "created_at >= ?";
        $bindTypes .= 's';
        $bindValues[] = $params['date_from'];
    }
    
    if ($params['date_to']) {
        $where[] = "created_at <= ?";
        $bindTypes .= 's';
        $bindValues[] = $params['date_to'] . ' 23:59:59';
    }
    
    if ($params['search']) {
        $where[] = "(transaction_id LIKE ? OR user_id LIKE ? OR payment_mode LIKE ?)";
        $bindTypes .= 'sss';
        $searchTerm = '%' . $params['search'] . '%';
        $bindValues[] = $searchTerm;
        $bindValues[] = $searchTerm;
        $bindValues[] = $searchTerm;
    }

    // Count total records
    $countSql = "SELECT COUNT(*) as total FROM transactions";
    if (!empty($where)) {
        $countSql .= " WHERE " . implode(" AND ", $where);
    }
    
    $countStmt = $db->prepare($countSql);
    if (!empty($bindValues)) {
        $countStmt->bind_param($bindTypes, ...$bindValues);
    }
    $countStmt->execute();
    $total = $countStmt->get_result()->fetch_assoc()['total'];

    // Fetch paginated data
    $sql = "
        SELECT 
            transaction_id, 
            user_id, 
            amount, 
            status, 
            payment_mode, 
            created_at,
            updated_at
        FROM transactions
    ";
    
    if (!empty($where)) {
        $sql .= " WHERE " . implode(" AND ", $where);
    }
    
    $sql .= " ORDER BY created_at DESC LIMIT ? OFFSET ?";
    $bindTypes .= 'ii';
    $bindValues[] = $params['limit'];
    $bindValues[] = ($params['page'] - 1) * $params['limit'];

    $stmt = $db->prepare($sql);
    $stmt->bind_param($bindTypes, ...$bindValues);
    $stmt->execute();
    $result = $stmt->get_result();

    // Format response
    $transactions = [];
    while ($row = $result->fetch_assoc()) {
        $transactions[] = [
            'id' => $row['transaction_id'],
            'user' => $row['user_id'],
            'amount' => (float)$row['amount'],
            'status' => $row['status'],
            'method' => $row['payment_mode'],
            'date' => $row['created_at'],
            'last_updated' => $row['updated_at']
        ];
    }

    echo json_encode([
        'success' => true,
        'data' => [
            'meta' => [
                'page' => (int)$params['page'],
                'limit' => (int)$params['limit'],
                'total' => (int)$total,
                'pages' => ceil($total / $params['limit'])
            ],
            'transactions' => $transactions
        ]
    ]);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'error' => $e->getMessage()
    ]);
} finally {
    $db->close();
}