<?php

include 'db.php';

// Get data from the request
$data = json_decode(file_get_contents('php://input'), true);
$userId = $data['userId'];
$technicianName = $data['technicianName'];
$technicianPhone = $data['technicianPhone'];
$uniqueBookingId = generateUniqueBookingId();

// Prepare SQL query to fetch confirmed bookings
$sql = "SELECT * FROM Service_booking WHERE user_id = :userId";
$stmt = $conn->prepare($sql);
$stmt->bindParam(':userId', $userId, PDO::PARAM_INT);

try {
    $stmt->execute();
    $result = $stmt->fetchAll(PDO::FETCH_ASSOC);

    if (empty($result)) {
        http_response_code(404);
        echo json_encode(["status" => "error", "message" => "No bookings found for this user."]);
        exit;
    }

    // Prepare the UPDATE statement
    $updateSql = "UPDATE Service_booking SET status = 'technician_assigned', technician_name = ?, technician_phone = ?, unique_booking_id = ? WHERE user_id = ?";
    $updateStmt = $conn->prepare($updateSql);

    // Execute the UPDATE statement
    if ($updateStmt->execute([$technicianName, $technicianPhone, $uniqueBookingId, $userId])) {

        $otp = generateOTP();

        // Hash and store OTP
        $hashedOtp = password_hash($otp, PASSWORD_DEFAULT);
        $otpSql = "UPDATE Service_booking SET otp = ? WHERE unique_booking_id = ?";
        $otpStmt = $conn->prepare($otpSql);
        $otpStmt->execute([$hashedOtp, $uniqueBookingId]);

        $isTestMode = true; // Set to true for testing, false for production

        if ($isTestMode) {
            echo json_encode(["status" => "success", "message" => "Technician assigned successfully with the OTP: " . $otp, "booking_id" => $uniqueBookingId]);
            exit;
        }

        // Production Logic (With SMS Gateway)
        $smsStatus = sendSMS($technicianPhone, "Your booking ID is: " . $uniqueBookingId . ". OTP for confirmation: " . $otp, $isTestMode);

        if ($smsStatus['status'] == 'success') {
            http_response_code(200);
            echo json_encode(["status" => "success", "message" => "Booking assigned and OTP sent.", "booking_id" => $uniqueBookingId]);
        } else {
            error_log("SMS sending error: " . $smsStatus['message']);
            http_response_code(200);
            echo json_encode(["status" => "success", "message" => "Booking assigned, but there was an issue sending the OTP. Check logs.", "booking_id" => $uniqueBookingId]);
        }

    } else {
        http_response_code(500);
        echo json_encode(["status" => "error", "message" => "Failed to update bookings"]);
    }

} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(["status" => "error", "message" => "Database error: " . $e->getMessage()]);
}

$conn = null;


function generateUniqueBookingId() {
    return uniqid();
}

function generateOTP($length = 6) {
    $characters = '0123456789';
    $otp = '';
    for ($i = 0; $i < $length; $i++) {
        $otp .= $characters[rand(0, strlen($characters) - 1)];
    }
    return $otp;
}

function sendSMS($phoneNumber, $message, $isTestMode) {
    if ($isTestMode) {
        return [
            'status' => 'success',
            'message' => 'SMS sent successfully (TEST MODE)',
        ];
    } else {
        $apiKey = 'YOUR_SMS_API_KEY'; // Replace with your API key
        $apiSecret = 'YOUR_SMS_API_SECRET'; // Replace with your API secret

        // ... (Your SMS API call logic here) ...

        $response =  [
            'status' => 'success', // Or 'error'
            'message' => 'SMS sent successfully', // Or error message
        ];

        return $response;
    }
}

?>